#include <bootcontrol.h>

/*
* This module works like a boot-script
* It sets the mmcpart environment variable depending on the environment variables BOOT_ORDER, BOOT_A_LEFT, BOOT_B_LEFT 
* and the ext_csd register of the eMMC. 
*/

int bootControl() {
	char* boxPrintBuf[5] = {NULL, NULL, NULL, NULL, NULL};
	char lineBuf[32] = {'\0'};

	puts("\n");
	printf(KGRN KBLD "Boot Control\n"KNRM);

	struct mmc* mmc = find_mmc_device(MMC_DEVICE);
	if (mmc_init(mmc) != 0) {
		printf(KRED"mmc initialization failed\n"KNRM);
		resetBoard(5);
	}

	/* get boot partition from ext_csd register and u-boot environment variables */
	u8 hwpart = EXT_CSD_EXTRACT_BOOT_PART(mmc->part_config);
	char* BOOT_A_LEFT = env_get(uvar_BOOT_A_LEFT);
	char* BOOT_B_LEFT = env_get(uvar_BOOT_B_LEFT);
	char* BOOT_ORDER = env_get(uvar_BOOT_ORDER);

	int iBOOT_A_LEFT;
	int iBOOT_B_LEFT;

	if (!(hwpart == MMCBLK2BOOT0 || hwpart == MMCBLK2BOOT1)) {
		/* selected bootpartition in ext_csd register is invalid */
		/* resetting the partition to boot0 despite the fact this bootloader */
		/* would not have been loaded if the hwpart was invalid */
		/* if something sets hwpart to 7, the bootloader will be read from mmcblk2 at address 0x8400 */
		printf(KRED"hwpart error: %d\n"KNRM, hwpart);
		mmc_set_part_conf(mmc, ACK, MMCBLK2BOOT0, ACCESS);
		
		resetBoard(0);
	}

	/* setting default values in case variables are not present */
	/* This is the case at the first boot after flashing */
	if (BOOT_A_LEFT == NULL) {
		printf(KYEL"BOOT_A_LEFT not found in environment. Setting BOOT_A_LEFT to default\n"KNRM);
		env_set(uvar_BOOT_A_LEFT, default_BOOT_A_LEFT);
		BOOT_A_LEFT = env_get(uvar_BOOT_A_LEFT);
	}
	if (BOOT_B_LEFT == NULL) {
		printf(KYEL"BOOT_B_LEFT not found in environment. Setting BOOT_B_LEFT to default\n"KNRM);
		env_set(uvar_BOOT_B_LEFT, default_BOOT_B_LEFT);
		BOOT_B_LEFT = env_get(uvar_BOOT_B_LEFT);
	}

	/* RAUC removes the target slot from the BOOT_ORDER when an installation is running and adds it after the installation again */
	if(strlen(BOOT_ORDER) == 1) {
		/* Update was interrupted when RAUC tried installing into B */
		if(strncmp(BOOT_ORDER, "A", 1) == 0) {
			printf(KRED"Update installation interrupted when installing into 'B'\n"KNRM);
			env_set(uvar_BOOT_ORDER, "A B");
			BOOT_ORDER = env_get(uvar_BOOT_ORDER);
		}

		/* Update was interrupted when RAUC tried installing into A */
		if(strncmp(BOOT_ORDER, "B", 1) == 0) {
			printf(KRED"Update installation interrupted when installing into 'A'\n"KNRM);
			env_set(uvar_BOOT_ORDER, "B A");
			BOOT_ORDER = env_get(uvar_BOOT_ORDER);
		}
	}

	/* set default BOOT_ORDER if BOOT_ORDER is null or neither A B nor B A */
	if (BOOT_ORDER == NULL ||  !(strncmp(BOOT_ORDER, "A B", strlen("A B")) == 0 || strncmp(BOOT_ORDER, "B A", strlen("B A")) == 0)) { 
		printf(KRED"BOOT_ORDER invalid. Setting BOOT_ORDER to default\n"KNRM);
		env_set(uvar_BOOT_ORDER, default_BOOT_ORDER);
		BOOT_ORDER = env_get(uvar_BOOT_ORDER);
	}

	/* strtoul returns 0 if string does not start with a numeric value */
	iBOOT_A_LEFT = (int)simple_strtoul(BOOT_A_LEFT, NULL, 10);
	iBOOT_B_LEFT = (int)simple_strtoul(BOOT_B_LEFT, NULL, 10);

	printf("BOOT_A_LEFT: %d\n", iBOOT_A_LEFT);
	printf("BOOT_B_LEFT: %d\n", iBOOT_B_LEFT);
	printf("BOOT_ORDER:  %s\n", BOOT_ORDER);
	printf("hwpart:      %s\n", hwpart == MMCBLK2BOOT0 ? "mmcblk2boot0" : (hwpart == MMCBLK2BOOT1 ? "mmcblk2boot1" : "unknown"));

	/* handling of boot order, left tries and used hwpart */
	if ((strcmp(BOOT_ORDER, "A B") == 0 && iBOOT_A_LEFT <= 0 && iBOOT_B_LEFT > 0 && hwpart == MMCBLK2BOOT0) ||
	    (strcmp(BOOT_ORDER, "A B") == 0 && iBOOT_A_LEFT <= 0 && iBOOT_B_LEFT > 0 && hwpart == MMCBLK2BOOT1) ||
		(strcmp(BOOT_ORDER, "B A") == 0 && iBOOT_A_LEFT  > 0 && iBOOT_B_LEFT > 0 && hwpart == MMCBLK2BOOT0) ||
		(strcmp(BOOT_ORDER, "B A") == 0 && iBOOT_A_LEFT <= 0 && iBOOT_B_LEFT > 0 && hwpart == MMCBLK2BOOT0)) {

		boxPrintBuf[0] = "FALLBACK!";
		boxPrintBuf[1] = "Setting BOOT_ORDER to B A and hwpart to boot1";
		printBox(boxPrintBuf, KRED, KRED);

		env_set(uvar_BOOT_ORDER, "B A");
		env_save();
		mmc_set_part_conf(mmc, ACK, MMCBLK2BOOT1, ACCESS);
		resetBoard(5);
	} 
	else
	if ((strcmp(BOOT_ORDER, "A B") == 0 && iBOOT_A_LEFT > 0 && iBOOT_B_LEFT >  0 && hwpart == MMCBLK2BOOT1) ||
	    (strcmp(BOOT_ORDER, "A B") == 0 && iBOOT_A_LEFT > 0 && iBOOT_B_LEFT <= 0 && hwpart == MMCBLK2BOOT1) ||
		(strcmp(BOOT_ORDER, "B A") == 0 && iBOOT_A_LEFT > 0 && iBOOT_B_LEFT <= 0 && hwpart == MMCBLK2BOOT0) ||
		(strcmp(BOOT_ORDER, "B A") == 0 && iBOOT_A_LEFT > 0 && iBOOT_B_LEFT <= 0 && hwpart == MMCBLK2BOOT1)) {

		boxPrintBuf[0] = "FALLBACK!";
		boxPrintBuf[1] = "Setting BOOT_ORDER to A B and hwpart to boot0";
		printBox(boxPrintBuf, KRED, KRED);

		env_set("BOOT_ORDER", "A B");
		env_save();
		mmc_set_part_conf(mmc, ACK, MMCBLK2BOOT0, ACCESS);
		resetBoard(5);
	} else
	if (iBOOT_A_LEFT <= 0 && iBOOT_B_LEFT <= 0) {
		boxPrintBuf[0] = "No tries left";
		boxPrintBuf[1] = "Setting up default state";
		boxPrintBuf[2] = "('A'-path boot)";
		printBox(boxPrintBuf, KRED, KRED);

		env_set(uvar_BOOT_ORDER, "A B");
		env_set(uvar_BOOT_A_LEFT, "3");
		env_set(uvar_BOOT_B_LEFT, "3");
		env_save();
		mmc_set_part_conf(mmc, ACK, MMCBLK2BOOT0, ACCESS);

		resetBoard(5);
	}

	/* --- first character in BOOT_ORDER is from now on valid --- */
	char BOOT_SLOT = BOOT_ORDER[0];
	char newBOOT_X_LEFT[2] = {'\0'};

	sprintf(lineBuf, "Selecting slot %c", BOOT_SLOT);
	boxPrintBuf[0] = lineBuf;
	printBox(boxPrintBuf, KGRN, KGRN);

	/* reset environment variables in case they are corrupted */
	char* vars[] = {"loadimage", "mmcdev", "img_addr", "bootdir", "image", "mmcargs", "mmcboot", "optargs"};
	int nvars = sizeof(vars)/sizeof(vars[0]);
	set_default_vars(nvars, vars, 0);

	if (BOOT_SLOT == 'A') {
		sprintf(newBOOT_X_LEFT, "%d", iBOOT_A_LEFT - 1);
		env_set("BOOT_A_LEFT", newBOOT_X_LEFT);
		env_set("mmcpart", "1");
		env_set("kernelargs", "rauc.slot=A");
		env_save();
	}

	if (BOOT_SLOT == 'B') {
		sprintf(newBOOT_X_LEFT, "%d", iBOOT_B_LEFT - 1);
		env_set("BOOT_B_LEFT", newBOOT_X_LEFT);
		env_set("mmcpart", "3");
		env_set("kernelargs", "rauc.slot=B");
		env_save();
	}

	return 0;
}

void resetBoard(int delaySec) {
	printf("Reset in %d seconds...\n", delaySec);
	udelay (delaySec * 1000 * 1000);
	disable_interrupts();
	reset_misc();
	reset_cpu(0);
}

void printBox(char* lines[], char* textColor, char* boxColor) {
	/* determine number of lines and length of longest line */
	int nLines = 0;
	int maxLength = 0;
	while(lines[nLines] != NULL) {
		int currentLineLength = strlen(lines[nLines]);
		maxLength = currentLineLength > maxLength ? currentLineLength : maxLength;
		nLines++;
	}

	/* expand box by one space on the left and on the right */
	maxLength += 2;

	/* print top box characters */
	printf("%s", boxColor);
	printf("┌");
	for(int i = 0; i < maxLength; i += 1) printf("─");
	printf("┐\n");
	printf("%s", KNRM);

	/* print content */
	for(int i = 0; i < nLines; i += 1) {
		int leftSpaces = (maxLength - strlen(lines[i])) / 2;
		int rightSpaces = maxLength - strlen(lines[i]) - leftSpaces;

		printf("%s", boxColor);
		printf("│");
		printf("%s", KNRM);

		for(int i = 0; i < leftSpaces; i += 1) printf(" ");
		/* print text color, line content and clear line in buffer */
		printf("%s%s%s", textColor, lines[i], KNRM); lines[i] = NULL;
		for(int i = 0; i < rightSpaces; i += 1) printf(" ");

		printf("%s", boxColor);
		printf("│\n");
		printf("%s", KNRM);
	}

	/* print bottom box characters */
	printf("%s", boxColor);
	printf("└");
	for(int i = 0; i < maxLength; i += 1) printf("─");
	printf("┘\n");
	printf("%s", KNRM);
}

